<?
class negtracker extends module {

    //
    // Author: Herman Tolentino (unless specified)
    //
    // AEFI NLP Project
    // Vaccine Analytic Unit - Brighton Collaboration - National Library of Medicine
    //
    // IMPORTANT:
    // This assumes that the UMLS Lexicon tables have been set up and that the LRAGR table
    // has been indexed using STR.
    //

    function negtracker() {
        //
        // do not forget to update version
        //
        $this->author = 'Herman Tolentino MD';
        $this->version = "0.1-".date("Y-m-d");
        $this->module = "negtracker";
        $this->description = "AEFI Module - Negation Tracker";
        // 0.1: BEGIN

    }

    function init_deps() {
    //
    // insert dependencies in module_dependencies
    //
        module::set_dep($this->module, "module");

    }

    function init_lang() {
    //
    // insert necessary language directives
    //
        module::set_lang("FTITLE_SPELLING_CHECK", "english", "SPELLING CHECK", "Y");
        module::set_lang("INSTR_SPELLING_CHECK", "english", "CLICK ON THE HIGHLIGHTED TERMS ABOVE", "Y");
        module::set_lang("LBL_POSSIBLE_MISPELLED", "english", "POSSIBLE MISPELLED WORD", "Y");
        module::set_lang("LBL_POSSIBLE_CORRECTIONS", "english", "POSSIBLE CORRECTIONS", "Y");
        module::set_lang("LBL_REPLACEMENT_TERM", "english", "REPLACEMENT TERM", "Y");
        module::set_lang("INSTR_SPELLCORRECTION_LIST", "english", "PICK ONE FROM THE LIST AND CLICK ON <b>Replace from Correction List</b>", "Y");
        module::set_lang("LBL_SPELLING_SOURCE", "english", "SPELLING SOURCE", "Y");
    }

    function init_menu() {
        if (func_num_args()>0) {
            $arg_list = func_get_args();
            $module_id = $arg_list[0];
        }

        module::set_menu($this->module, "Negation Modifiers", "LIBRARIES", "_negation_modifiers");

        // put in more details
        module::set_detail($this->description, $this->version, $this->author, $this->module);
    }

    function init_stats() {
    }

    function init_help() {
    }

    function init_sql() {
        if (func_num_args()>0) {
            $arg_list = func_get_args();
            $module_id = $arg_list[0];
        }

        module::execsql("CREATE TABLE `m_negtracker_modifiers` (".
            "`modifier_id` float NOT NULL auto_increment,".
            "`modifier_str` varchar(150) NOT NULL default '',". // Put it in as regular expression
            "`modifier_known_tags` varchar(100) default '',".   // what other tags does it modify?
            "`modifier_tag_context` varchar(100) default '',".  // how is modifier tagged?
            "`modifier_type` varchar(10) default '',".             // bidirectional, left, right
            "`modifier_algebra` char(1) default 'N',".          // algebraic pattern? no nausea and/or vomiting
            "`modifier_example` varchar(255) default '',".      // example of usage for this modifier
            "PRIMARY KEY  (`modifier_id`), ".
            "UNIQUE INDEX (`modifier_str`,`modifier_type`), ".
            "INDEX (`modifier_str`)".
            ") ENGINE=MyISAM");

        module::execsql("CREATE TABLE `m_negtracker_inventory` (".
            "`document_id` varchar(100) NOT NULL,".
            "`modifier_str` varchar(150) NOT NULL,".
            "`modifier_type` varchar(10) default '',".
            "`tag_context` varchar(150) NOT NULL,".
            "`frequency` integer NOT NULL default '0',".
            "PRIMARY KEY (`document_id`, `modifier_str`, `tag_context`)".
            ") ENGINE=MyISAM");

    }

    function drop_tables() {

        module::execsql("DROP TABLE `m_negtracker_modifiers`;");
        module::execsql("DROP TABLE `m_negtracker_inventory`;");
    }


    // --------------- CUSTOM MODULE FUNCTIONS ------------------

    function negated_concepts() {
        if (func_num_args()) {
            $arg_list = func_get_args();
            $inputstr = $arg_list[0];

            // NO
            $patterns[0] = "/\b([Nn]o_D[DB])(\s?)([a-zA-Z]+_[A-Z]{2,4}_C[0-9]{7})/";
            $patterns[1] = "/\b([Nn]o_D[DB])(\s?)([a-zA-Z]+_JJ)(\s?)([a-zA-Z]+_[A-Z]{2,4}_C[0-9]{7})/";
            $patterns[2] = "/\b([Nn]o_D[DB])(\s?)([a-zA-Z]+_DB)(\s?)([a-zA-Z]+_[A-Z]{2,4}_C[0-9]{7})/";

            // UNLIKELY
            $patterns[3] = "/\b([Uu]nlikely_D[DB])(\s?)/";

            $sentences = preg_split("/(\._\.)|(!_\.)|(\?_\.)/", $inputstr);
            for ($i=0; $i<count($sentences); $i++) {
                for ($j=0; $j<count($patterns); $j++) {
                    $matches = array();
                    if (preg_match_all($patterns[$j], $sentences[$i], $matches)) {
                        //$negated_concepts[] = $sentences[$i];
                        $line_list = implode("\n", $matches[0]);
                        $negated_concepts[] = $line_list;
                    }
                }
            }
            if (count($negated_concepts)>0) {
                return implode("\n",$negated_concepts);
            }
        }
    }

    function _negation_modifiers() {
        if (func_num_args()) {
            $arg_list = func_get_args();

            $menu_id = $arg_list[0];
            $get_vars = $arg_list[1];
            $post_vars = $arg_list[2];
            $textlab = $arg_list[3];

            if ($post_vars["submitdict"]) {
                negtracker::process_manage_modifier($menu_id, $get_vars, $post_vars, $textlab);
            }
            negtracker::form_manage_modifier($menu_id, $get_vars, $post_vars, $textlab);
        }
    }

    function process_manage_modifier() {
        if (func_num_args()) {
            $arg_list = func_get_args();
            $menu_id = $arg_list[0];
            $get_vars = $arg_list[1];
            $post_vars = $arg_list[2];
            $textlab = $arg_list[3];

            switch ($post_vars["submitdict"]) {
            case "Add Modifier":
                break;
            case "Update Modifier":
                break;
            case "Delete Modifier":
                break;
            case "Cancel":
                break;
            }
        }
    }

    function form_manage_modifier() {
        if (func_num_args()) {
            $arg_list = func_get_args();
            $menu_id = $arg_list[0];
            $get_vars = $arg_list[1];
            $post_vars = $arg_list[2];
            $textlab = $arg_list[3];

            $term = $get_vars["spellcheck"];

            print "<a name='spellcheck'>";
            print "<span class='newstitle'>".FTITLE_SPELLING_CHECK."</span><br>";
            print "<span class='tinylight'>".INSTR_SPELLING_CHECK."</span><br>";
            print "<table width='450' cellpadding='5' cellspacing='1' style='border: 1px solid black'>";
            print "<form action = '".$_SERVER["SELF"]."?page=PROCESSING&menu_id=".$get_vars["menu_id"].($get_vars["document_id"]?"&document_id=".$get_vars["document_id"]:"")."&tab=".$get_vars["tab"]."&spellcheck=".$get_vars["spellcheck"]."#spellcheck' name='form_manage_term' method='post'>";
            print "<tr valign='top'><td>";
            print "<table cellpadding='2' cellspacing='1'>";
            print "<tr><td>";
            print "<span class='boxtitle'>".LBL_SPELLING_SOURCE."</span><br> ";
            $post_vars["spellsource"] = (isset($post_vars["spellsource"])?$post_vars["spellsource"]:"L"); // Lexicon is default source
            print "<input type='radio' name='spellsource' value='P' ".($post_vars["spellsource"]=="P"?"checked":"")."'> PSpell<br>";
            print "<input type='radio' name='spellsource' value='L' ".($post_vars["spellsource"]=="L"?"checked":"")."'> UMLS Lexicon<br>";
            print "</td></tr>";
            print "<tr><td>";
            print "<span class='boxtitle'>".LBL_POSSIBLE_MISPELLED."</span><br> ";
            print "<input type='text' class='textbox' size='20' maxlength='20' name='misspelled' value='".($term?$term:$post_vars["misspelled"])."' style='border: 1px solid #000000'><br>";
            print "</td></tr>";
            print "<tr><td>";
            print "<span class='boxtitle'>".LBL_POSSIBLE_CORRECTIONS."</span><br> ";
            if ($word_list = spelling::word_list($menu_id, $post_vars, $get_vars, $term)) {
                print "<span class='tinylight'>".INSTR_SPELLCORRECTION_LIST."</span><br>";
                print "<select size='10' name='correction_list'>";
                foreach ($word_list as $key=>$value) {
                    print "<option value='$value'>".$value."</option>";
                }
                print "</select>";
            } else {
                print "<font color='red'>None</font>";
            }
            print "</td></tr>";
            print "<tr><td><br>";
            print "<span class='boxtitle'>".LBL_REPLACEMENT_TERM."</span><br> ";
            print "<input type='text' class='textbox' size='20' maxlength='20' name='replacement' value='".($term?$term:$post_vars["misspelled"])."' style='border: 1px solid #000000'><br>";
            print "</td></tr>";
            print "</table>";
            print "</td><td>";
            print "<table cellpadding='2' cellspacing='1'>";
            print "<tr><td>";
            print "<input type='hidden' name='spellcheck' value='".$get_vars["spellcheck"]."'>";
            print "<input type='hidden' name='cleaned_text' value='".$textlab["cleaned"]."'>";
            print "<input type='submit' value = 'Change Dictionary' class='textbox' name='submitdict' title='Changes dictionary used for spelling check.' style='border: 1px solid #000000'> ";
            print "</td></tr>";
            print "<tr><td>";
            print "<input type='submit' value = 'Add to Dictionary' class='textbox' name='submitdict' title='Adds to the dictionary the word from the MISSPELLED WORD field.' style='border: 1px solid #000000'> ";
            print "</td></tr>";
            print "<tr><td>";
            print "<input type='submit' value = 'Replace from Correction List' class='textbox' name='submitdict' title='Replaces misspelled word with word selected from the CORRECTION LIST.' style='border: 1px solid #000000'> ";
            print "</td></tr>";
            print "<tr><td>";
            print "<input type='submit' value = 'Replace using Replacement Term' class='textbox' name='submitdict' title='Replaces misspelled word with word selected from the REPLACEMENT TERM field.' style='border: 1px solid #000000'> ";
            print "</td></tr>";
            print "<tr><td>";
            print "<input type='submit' value = 'Add to Ignore List' class='textbox' name='submitdict' title='Adds the word in the MISPELLED WORD field to the Ignore list.' style='border: 1px solid #000000'> ";
            print "</td></tr>";
            print "<tr><td>";
            print "<input type='submit' value = 'Cancel' class='textbox' name='submitdict' style='border: 1px solid #000000'> ";
            print "</td></tr>";
            print "</table>";
            print "</td></tr>";
            print "</form>";
            print "</table>";
        }
    }

}

/** ************************************************
** NEGATION CLASS
** $Author: Herman Tolentino MD
** $Last update: 10/02/05
**
** OBJECTIVE: Process one document at a time for
** negation phrases that affect concepts.
**
** Word list for negation and source code ideas were
** obtained from the work of Wendy Chapman using
** Negex (written in Python).
**
*/
class negation {

    var $inputstring;
    var $sentences;
    var $deep_parse;
    var $array_nega;
    var $array_negb;
    var $array_pnega;
    var $array_pnegb;
    var $array_conj;
    var $array_pseudoneg;
    var $negphrase;
    var $negphrase_collection;
    var $negation_stats;

    function negation($inputstring) {

        // initialize negation statistics array
        $this->negation_stats["sentence_count"] = 0;
        $this->inputstring = $inputstring;
        $this->partition();
        $this->catch_phrase();
        $this->load_dict();
        $this->negphrase_collection = $this->scrutinize();
    }

    function partition() {
        $this->sentences = preg_split("/(\._\.)|(!_\.)|(\?_\.)|(-_SYM)|(-_:)/", $this->inputstring);
        $this->negation_stats["sentence_count"] = count($this->sentences);
    }

    function get_sentence_count() {
        return $this->negation_stats["sentence_count"];
    }

    function get_negation_phrases() {
        foreach ($this->negphrase_collection as $key=>$value) {
            if (strlen(trim($value))==0) {
                unset($this->negphrase_collection[$key]);
            }
        }
        $this->negation_stats["collection"] = $this->negphrase_collection;
        return $this->negation_stats["collection"];
    }

    function catch_phrase() {

        $pattern[0] = "/\b(absence_)/";
        $pattern[1] = "/\b(cannot_)/";
        $pattern[2] = "/\b(decline_)/";
        $pattern[3] = "/\b(den)(y|ies|ied)(_)/";
        $pattern[4] = "/\b(fail_)/";
        $pattern[5] = "/\b(free_)/";
        $pattern[6] = "/\b(negative_)/";
        $pattern[7] = "/\b(never_)/";
        $pattern[8] = "/\b(not|[Nn]o)(_)/";
        $pattern[9] = "/\b(with_|without_)(\w+)(no_)/";
        $pattern[10] = "/\b(rule[sd]_)/";
        $pattern[11] = "/\b(unlikely_)/";
        for($i=0; $i<count($this->sentences); $i++) {
            for($j=0; $j<count($pattern); $j++) {
                $matches = array();
                if (preg_match_all($pattern[$j], $this->sentences[$i], $matches)) {
                    $this->deep_parse[] = $this->sentences[$i];
                }
            }
        }
        if (is_array($this->deep_parse)) {
            $this->deep_parse = array_unique($this->deep_parse);
            $this->negation_stats["candidate_count"] = count($this->deep_parse);
        }

    }

    function get_candidate_count() {
        return $this->negation_stats["candidate_count"];
    }

    function load_dict() {

        $sql = "select modifier_str, modifier_type from m_negtracker_modifiers order by modifier_type";
        if ($result = mysql_query($sql)) {
            if (mysql_num_rows($result)) {
                while (list($modifier, $type) = mysql_fetch_array($result)) {
                    switch ($type) {
                    case "NEGA":
                        $this->array_nega[] = $modifier;
                        break;
                    case "NEGB":
                        $this->array_negb[] = $modifier;
                        break;
                    case "PNEGA":
                        $this->array_pnega[] = $modifier;
                        break;
                    case "PNEGB":
                        $this->array_pnegb[] = $modifier;
                        break;
                    case "PSEUDONEG":
                        $this->array_pseudoneg[] = $modifier;
                        break;
                    case "CONJ":
                        $this->array_conj[] = $modifier;
                        break;
                    }
                }
            }
        }
    }

    function scrutinize() {

        // LEVEL ONE
        foreach($this->deep_parse as $key => $sentence) {
            $temp = preg_split("/(\s+|,_,)/", $sentence);
            for($i=0; $i<count($temp); $i++) {
                if (strlen($temp[$i])>0) {
                    $tokens[] = $temp[$i];
                }
            }
            $stripped = $this->remove_tags($sentence);
            $phrase = new phrase($stripped);
            $phrase_array = $phrase->detect($phrase->get_sentences());
            print_r($phrase->get_phrase_terms());
            $this->negation_stats["stripped_sentences"][] = $stripped;
            for($i=0; $i<count($tokens); $i++) {
                $mytoken = new token($tokens[$i]);
                $term = $mytoken->get_term();
                $tag = $mytoken->get_tag();
                $cui = $mytoken->get_cui();
                // look up term if is a negation modifier
                if ($negtype = $this->is_negation($term)) {
                    // if it is a negation modifier,
                    $negphrase = $this->negation_phrase($sentence, $term, $negtype);
                    if (is_array($negphrase)) {
                        foreach($negphrase as $value) {
                            $this->negphrase[] = $value;
                        }
                    }
                }
            }
        }
        if (is_array($this->negphrase)) {
            $this->negation_stats["negation_phrases"] = array_unique($this->negphrase);
            return $this->negation_stats["negation_phrases"];
        }
    }

    function get_stripped_sentences() {
        return $this->negation_stats["stripped_sentences"];
    }

    function remove_tags($sentence) {

        $tokens = explode(" ", $sentence);
        for($i=0; $i<count($tokens); $i++) {
            $token = new token($tokens[$i]);
            $stripped[] = $token->get_term();
        }
        return implode(" ", $stripped);
    }

    function sentence_split($sentence) {
    //
    // split sentence on conjunctions
    //
        foreach($this->array_conj as $conj) {
            if ($phrase = preg_split("/(".$conj.")(_)([A-Z]{0,3})/")) {
                return $phrase;
            }
        }
    }

    function regex($term) {

        $first = substr($term, 0,1);
        $ucfirst = substr(ucfirst($term), 0,1);
        $rest = substr($term, 1, strlen($term)-1);
        $regex = "[".$ucfirst.$first."]".$rest;
        return $regex;
    }

    function negation_phrase($sentence, $term, $negtype) {
    //
    // this is a deep parse function
    //
    // NEGA negation comes before term modified
    // NEGB negation comes after term modififed

        // sequentially determine of sentence can be
        // split further using conjunctions, punctuations
        if ($this->is_stop($sentence)) {
            $phrases[] = sentence_split($sentence);
        } elseif (preg_match("/(,_,)/", $sentence)) {
            $phrases = preg_split("/(,_,)/", $sentence);
        } else {
            $phrases[] = $sentence;
        }
        // iterate through each phrase
        foreach ($phrases as $phrase) {
            switch($negtype) {
            case "NEGA":
                $tokens = explode(" ", $phrase);
                $negphrase = array();
                $start = false;
                for($i=0; $i<count($tokens); $i++) {
                    if (preg_match("/(to_II)/", $tokens[$i])) {
                        break;
                    }
                    if (preg_match("/\b(".$this->regex($term)."_)/", $tokens[$i])||$start==true) {
                        $start = true;
                        if (strlen($tokens[$i])>0) {
                            $negphrase[] = stripslashes($tokens[$i]);
                        }
                    }
                }
                // negation algebra stuff
                $negphrase = $this->semantic_processing($term, $negphrase, $negtype);
                if (is_array($negphrase)) {
                    $retval[] = implode(" ", $negphrase);
                }
                break;

            case "NEGB":
                break;
            case "PNEGA":
                break;
            case "PNEGB":
                break;
            }
        }
        return $retval;
    }

    function track_negphrase($negation_term, $negation_target) {
        $track = array("modifier"=>$negation_term, "target"=>$negation_target);
        $this->negation_stats["negation_phrase"][] = $track;
    }

    function get_track_negphrase() {
        array_unique($this->negation_stats["negation_phrase"]);
        return $this->negation_stats["negation_phrase"];
    }

    function semantic_processing($term, $negphrase, $negtype) {

        $aefi = new aefi();
        $filtered = array();

        for($i=0; $i<count($negphrase); $i++) {
            if (preg_match("/\b(".$this->regex($term).")_/", $negphrase[$i])) {
                $negation_term = $negphrase[$i];
            }
            if (count($negphrase)==4 && preg_match("/(or_CC)/", $negphrase[$i])) {
                if ($aefi->is_ssx($negphrase[$i-1]) && $aefi->is_ssx($negphrase[$i+1])) {
                    $filtered[] = $negation_term." ".$negphrase[$i-1];
                    $this->track_negphrase($negation_term, $negphrase[$i-1]);
                    $filtered[] = $negation_term." ".$negphrase[$i+1];
                    $this->track_negphrase($negation_term, $negphrase[$i+1]);
                    break;
                }
            }
        }
        if (is_array($filtered)) {
            array_unique($filtered);
        }
        if ($filtered) {
            return $filtered;
        } else {
            return $negphrase;
        }
    }

    function is_stop($term) {
        if (array_search($term, $this->array_conj)) {
            return true;
        }
        return false;
    }

    function is_negation($term) {

        if (array_search($term, $this->array_nega)) {
            return "NEGA";
        }
        if (array_search($term, $this->array_negb)) {
            return "NEGB";
        }
        if (array_search($term, $this->array_pnega)) {
            return "PNEGA";
        }
        if (array_search($term, $this->array_pnegb)) {
            return "PNEGB";
        }
        if (array_search($term, $this->array_pseudoneg)) {
            return "PSEUDONEG";
        }
        return false;
    }
}

/** ************************************************
** RBNEG CLASS
** $Author: Herman Tolentino MD
** $Last update: 02/18/06
**
**
*/
class semantic_negation {

    var $cuitext;
    var $phrasetext;
    var $tokens;
    var $rules;
    var $semantic_types;
    var $concept_separators;
    var $stop_words;

    private $bigrams;

    function semantic_negation($cuitext, $phrasetext) {
        $this->cuitext = $cuitext;
        $this->phrasetext = $phrasetext;
        $this->tokenize();
        $this->semantic_types = array("T042", "T061", "T078", "T169", "T046", "T047", "T184", "T080", "T121", "T033", "T040", "T181", "T037", "T023", "T030", "T109", "T081", "T041");
        $this->concept_separators = array("or_CC", ",_,", "and_CC", "\/_SYM");
        $this->stop_words = array("to_II", "._.");
        $this->process_tokens();
    }

    function tokenize() {
        $this->tokens = preg_split("/(\s+)/", $this->cuitext);
        $this->phrasetokens = preg_split("/(\s+)/", $this->phrasetext);
    }

    function process_tokens() {

        $negation_phrase = array();

        $state = "STOP";
        for ($i=0; $i<count($this->tokens)-1; $i++) {
            if (preg_match("/([Nn][Oo]_DD_C1298908 mention_NN of_II_C0332285|[Nn][Oo]_DD_C1298908|nor_CC|[Nn]either_CC|[Nn]ot_RR|[Ww]ithout_II_C0332288|([Rr]uled_VVN_C0870077)|[Dd]enies_(VVZ|NNS)_C0332319)/",$this->tokens[$i]) || $state == "START") {
                array_unshift($negation_phrase, $this->tokens[$i]);
                $state = "START";
                $token = new token($this->tokens[$i]);
                if (preg_match("/(,_,|_II|_\.|-_SYM|_PN|_TIMEU|_TO|but_CC|_MC|_:|during_II_C0347984|at_(II|RR\+)|in_II(_C0332285|)|after_(II|CS)(_C0231290|))/", $this->tokens[$i]) && !$token->_is_cui_tagged($this->tokens[$i])) {
                    array_unshift($negation_phrase, $this->tokens[$i]);
                    $state = "STOP";
                    //print_r($negation_phrase);
                    $negation_phrase = array();
                }
                if ($token->get_cui() && $state=="START") {
                    $semantic_type = $this->get_semantic_type($token->get_cui());
                    reset($this->semantic_types);
                    if (in_array($semantic_type, $this->semantic_types)) {
                        print $state." ".$this->tokens[$i]." ".$semantic_type."<br>";
                        array_unshift($negation_phrase, $this->tokens[$i]);
                        $state = "START";
                    }
                } else {
                    print $state." ".$this->tokens[$i]."<br>";
                }
            }
        }
    }

    function get_semantic_type($cui) {

        $sql = "select TUI from umls.MRSTY where CUI = '$cui'";
        if ($result = mysql_query($sql)) {
            if (mysql_num_rows($result)) {
                if (list($tui) = mysql_fetch_array($result)) {
                    return $tui;
                }
            }
        }
    }

    // end of class
}

/** ************************************************
** NEGPARSE CLASS
** $Author: Herman Tolentino MD
** $Last update: 11/28/05
**
** This is the Finite State Machine negation parser.
**
*/
class negparse {

    var $inputstring;
    var $phrasetokens;
    var $sentences;
    var $sentence_count;
    var $candidate_parse;
    var $state;
    var $negated_concepts;
    var $stack;

    function negparse($inputstring, $inputphrases) {

        $this->negated_concepts = array();
        $this->stack = array();
        $this->load_dict();
        $this->inputstring = $inputstring;
        $this->phrasetokens = $this->partition_phrases($inputphrases);
        $this->sentences = $this->partition();
        for ($i=0; $i<count($this->sentences); $i++) {
            $this->catch_modifiers($this->sentences[$i]);
        }
    }

    function partition_phrases($inputphrases) {

        $tokens = preg_split("/(\s+)/", $inputphrases);
        foreach ($tokens as $token) {
            $fragments = preg_split("/(_)/", $token);
            $cui = $fragments[2];
            $terms = preg_split("/(\|)/", $fragments[0]);
            $phrasetokens[] = array("phrase"=>$fragments[0], "terms"=>$terms, "cui"=>$cui);
        }
        return $phrasetokens;
    }

    function load_dict() {

        $sql = "select modifier_str, modifier_type from m_negtracker_modifiers order by modifier_type";
        if ($result = mysql_query($sql)) {
            if (mysql_num_rows($result)) {
                while (list($modifier, $type) = mysql_fetch_array($result)) {
                    switch ($type) {
                    case "NEGA":
                        $this->array_nega[] = $modifier;
                        break;
                    case "NEGB":
                        $this->array_negb[] = $modifier;
                        break;
                    case "PNEGA":
                        $this->array_pnega[] = $modifier;
                        break;
                    case "PNEGB":
                        $this->array_pnegb[] = $modifier;
                        break;
                    case "PSEUDONEG":
                        $this->array_pseudoneg[] = $modifier;
                        break;
                    case "CONJ":
                        $this->array_conj[] = $modifier;
                        break;
                    }
                }
            }
        }
    }

    function is_stop($input) {

        $token = new token($input);
        if (in_array($token->term, $this->array_conj)) {
            return $token->term;
        }
        return false;
    }

    function partition() {
        return preg_split("/(\._\.)|(!_\.)|(\?_\.)|(-_SYM)|(-_:)/", $this->inputstring);
    }

    function catch_modifiers($sentence) {

        $pattern[0] = "/\b(absence_)/";
        $pattern[1] = "/\b(cannot_)/";
        $pattern[2] = "/\b(decline_)/";
        $pattern[3] = "/\b(den)(y|ies|ied)(_)/";
        $pattern[4] = "/\b(fail_)/";
        $pattern[5] = "/\b(free_)/";
        $pattern[6] = "/\b(negative_)/";
        $pattern[7] = "/\b(never_)/";
        $pattern[8] = "/(.*)([Nn]o)(_)(.*)/";
        $pattern[9] = "/\b(with_|without_)(\w+)(no_)/";
        $pattern[10] = "/\b(rule[sd]_)/";
        $pattern[11] = "/\b(unlikely_)/";

        $patterns[0] = "/(.*)([Nn]o_DD_C1298908)(.*)(\s)([a-z]+_(II|TO|VV))(.*)/";
        $patterns[1] = "/(.*)([Nn]o_DD_C1298908)(\s)([A-Za-z]+_[VGNSJ]+_C[0-9]{7})(.*)/";
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $sentence)) {
                $this->deep_parse($sentence, $pattern, "NO");
            }
        }

        $patterns[0] = "/(.*)([Dd]enies_VVZ_C0332319)(.*)/";
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $sentence)) {
                $this->deep_parse($sentence, $pattern, "DENY");
            }
        }
    }

    function deep_parse($sentence, $pattern, $parsetype) {
    // FSM parser

        static $parse_tree;

        $this->state = "START";
        print "<font color='blue'>PARSING: ".$sentence."</font><br>";
        while ($this->state<>"EOS") {

            print "STATE: <font color='red'>".$this->state."</font><br>";
            switch ($this->state) {
            case "START":

                switch ($parsetype) {
                case "NO":
                    $this->state = "VP1_NO";
                    break;
                case "DENY":
                    $this->state = "VP1_DENY";
                    break;
                }
                break;

            case "VP1_DENY":
                $this->state = "EOS";
                $step1 = preg_replace($pattern, "$3", $sentence);
                break;

            case "VP1_NO":
                // SIMPLE NEGATION
                $this->state = "EOS";
                if (preg_match("/([Nn]o_DD_C1298908)(\s)(further_JJ)/", $sentence)) {
                    print "NEGATION AMBIGUITY DETECTED<br>";
                    $this->state = "EOS";
                    break;
                }
                $matches = array();
                if (preg_match("/([Nn]o_DD_C1298908)(\s)([A-Za-z]+_[RTJ]+[_C0-9]{0,10})(\s)([A-Za-z]+_[VGNSJ]+_C[0-9]{7})/", $sentence, $matches)) {
                    $parse_array = $matches;
                    $this->state = "VP1_NO_PHR2";
                } else {
                    $this->state = "VP2_NO";
                }
                break;

            case "VP2_NO":
                $matches = array();
                if (preg_match("/([Nn]o_DD_C1298908)(\s)([A-Za-z]+_[VGNSJ]+_C[0-9]{7})/", $sentence, $matches)) {
                    $parse_array = $matches;
                    $this->state = "VP1_NO_SN";
                } else {
                    $this->state = "VP3_NO";
                }
                break;

            case "VP3_NO":
                $matches = array();
                if (preg_match("/([Nn]o_DD_C1298908)(\s)([A-Za-z]+_[VGNSJ]+_C[0-9]{7})(\s{0,1})(,_,)/", $sentence, $matches)) {
                    $this->add_stack($sentence, "VP1_NO_LIST");
                    //$this->state = "EOS";
                    $this->state = "VP1_NO_LIST";
                } else {
                    $this->state = "VP4_NO";
                }
                break;

            case "VP4_NO":
                $matches = array();
                if (preg_match("/([Nn]o_DD_C1298908)(\s)([A-Za-z]+_N[NS]+_C[0-9]{7})(and_CC)(\s)([A-Za-z]+_N[NS]+_C[0-9]{7})/", $sentence, $matches)) {
                    $this->negated_concepts = array_merge($this->process_tokens($matches), $this->negated_concepts);
                    print "bingo1";
                    $this->state = "EOS";
                }
                $matches = array();
                if (preg_match("/([Nn]o_DD_C1298908)(\s)([A-Za-z]+_J[JRT]+[_C0-9]{0,10})(\s)([A-Za-z]+_N[NS]+_C[0-9]{7})/", $sentence, $matches)) {
                    $this->negated_concepts = array_merge($this->process_tokens($matches), $this->negated_concepts);
                    print "bingo2";
                    $this->state = "EOS";
                }
                $this->state = "EOS";
                break;

            case "VP1_NO_SN":
                $this->negated_concepts = array_merge($this->process_tokens($parse_array), $this->negated_concepts);
                array_unique($this->negated_concepts);
                $this->state = "VP3_NO";
                break;

            case "VP1_NO_LIST":
                // assume there is a list after NO
                $negated = $this->process_list($sentence);
                $this->negated_concepts = array_merge($this->negated_concepts, $negated);
                array_unique($this->negated_concepts);
                $this->state = "VP4_NO";
                break;

            case "VP1_NO_PHR":
                // assume there are noun phrases
                $negated = $this->process_phrase($sentence);
                $this->negated_concepts = array_merge($this->negated_concepts, $negated);
                array_unique($this->negated_concepts);
                $this->state = "EOS";
                break;

            case "VP1_NO_PHR2":
                $matches = array();
                if (preg_match("/([Nn]o_DD_C1298908)(\s+)([A-Za-z]+_J[JRT_C0-9]+)(\s+)([A-Za-z]+_N[NS]+_C[0-9]{7})/", $sentence, $matches)) {
                    $negated = $this->process_phrase($matches[0]);
                    $this->negated_concepts = array_merge($this->negated_concepts, $negated);
                    array_unique($this->negated_concepts);
                }
                $this->state = "VP2_NO";
                break;
            }


        }
        $this->lookback();
        print "STATE: <font color='red'>".$this->state."</font><br><br>";

    }

    function lookback() {
    }

    function is_in_stack ($sentence, $state) {

        print_r($this->stack);
        foreach ($this->stack as $item) {
            if ($item["sentence"] == $sentence && $item["state"] == $state) {
                return true;
            }
        }
        return false;
    }

    function add_stack ($sentence, $state) {

        $item[] = array("sentence" => $sentence, "state"=>$state);
    }

    function is_negated_concept($cui) {

        foreach($this->negated_concepts as $concept) {
            if (trim($concept)==trim($cui)) {
                return true;
            }
        }
        return false;
    }

    function process_phrase($sentence) {

        $tokens = preg_split("/(\s+)/", $sentence);
        for ($i=0; ($i<count($tokens) || $i<=5); $i++) {
            if ($cui = $this->match_phrase($tokens, $i)) {
                $negated[] = $cui;
            }
            if ($stop = $this->is_stop($tokens[$i])) {
                print "STOP WORD: $stop<br>";
                break;
            }
            if (preg_match("/(to_(TO|II))/", $tokens[$i])) {
                print "STOP WORD: ".$tokens[$i]."<br>";
                break;
            }
        }
        if (is_array($negated)) {
            $negated = array_unique($negated);
            foreach ($negated as $value) {
                print "PHRASE CUI: $value<br>";
            }
        }
        return $negated;
    }

    function match_phrase($tokens, $index) {
    //
    // process tokens in a sentence
    //
        for($i=0; $i<count($this->phrasetokens); $i++) {
        // search through phrase tokens identified in document

            $score[$i]["phrase"] = $this->phrasetokens[$i]["phrase"];
            $score[$i]["score"] = 0;
            $score[$i]["max"] = count($this->phrasetokens[$i]["terms"]);
            $score[$i]["cui"] = $this->phrasetokens[$i]["cui"];

            for ($j=$index; ($j < 5 || $j < count($tokens)); $j++) {
            // iterate through tokens in sentence up to 5 tokens (window)
            // starting from index
                if (isset($tokens[$j])) {
                    $dtoken = new token($tokens[$j]);
                }
                if (in_array($dtoken->term, $this->phrasetokens[$i]["terms"])) {
                    //print "PHRASE: ".implode(" ", $this->phrasetokens[$i]["terms"])."<br>";
                    //print "FOUND PHRASE TOKEN: ".$dtoken->term."<br>";
                    $score[$i]["score"] = (float) $score[$i]["score"] + 1;
                }
                if ($score[$i]["score"]>0) {
                    if ($score[$i]["score"] == $score[$i]["max"]) {
                        return $score[$i]["cui"];
                    }
                }
            }
        }

    }

    function process_list($sentence) {

        $sentence = preg_replace("/([Nn]o_DD_C1298908)/", "", $sentence);
        print_r($sentence);
        $fragments = preg_split("/(,_,)|(\s+)|(or_CC)/", $sentence);
        $i = 0;
        foreach ($fragments as $fragment) {
            //print "FRAGMENT: ".$fragment."<br>";
            $i++;
            if ($i>17) {
                break;
            }
            $fragment = trim($fragment);
            $token = new token($fragment);
            $aefi = new aefi();
            if ($aefi->is_ssx($token->cui)) {
                print "NEGATED CUI: ".$token->cui."<br>";
                $negated[] = $token->cui;
            }
            if ($stop = $this->is_stop($fragment)) {
                print "STOP WORD: $stop<br>";
                break;
            }
        }
        if (is_array($negated)) {
            $negated = array_unique($negated);
        }
        return $negated;
    }

    function process_tokens($matches) {

        for ($i=1; ($i<count($matches)||$i<=5); $i++) {
            if (!preg_match("/([Nn]o_DD_C1298908)/", $matches[$i])) {
                if (strlen(trim($matches[$i]))>0) {
                    $token = new token($matches[$i]);
                    if (strlen(trim($token->cui))>0) {
                        print "NEGATED CUI: ".$token->cui."<br>";
                        $aefi = new aefi();
                        if ($aefi->is_ssx($token->cui)) {
                            $negated[] = $token->cui;
                        }
                    }
                }
            }
        }
        if (is_array($negated)) {
            $negated = array_unique($negated);
        }
        return $negated;
    }


}

class rules {

    var $rule_db;

    function rules($inputstring) {
        $this->rule_db = array();
        $this->inputstring = $inputstring;
    }

    function scan($definition) {
        for($i=0; $i<count($this->rule_db); $i++) {

        }
    }

    function add_tag_rule($definition, $tag) {
        $this->rule_db[] = array("type"=>"tag", "definition"=>$definition, "value"=>$tag);
    }

    function add_concept_rule($definition, $concept) {
        $this->rule_db[] = array("type"=>"concept", "definition"=>$definition, "value"=>$concept);
    }

    function add_term_rule($definition, $term) {
        $this->rule_db[] = array("type"=>"term", "definition"=>$definition, "value"=>$term);
    }

}

/** ************************************************
** NEGATION MANAGER CLASS
** $Author: Herman Tolentino MD
** $Last update: 10/05/05
*/
class negation_manager {

    var $arglist;

    function negation_manager() {

        if (func_num_args()) {
            $arg_list = func_get_args();
            $menu_id = $arg_list[0];
            $post_vars = $arg_list[1];
            $get_vars = $arg_list[2];
            $textlab = $arg_list[3];
            $this->arglist = $arg_list;
        }

        if (isset($post_vars["submitcui"])) {
            $this->process_manage_negation();
        }
        $this->form_manage_negation();
    }

    function mark_tagged($term, $text, $get_vars) {

        $arg_list = $this->arglist;
        $menu_id = $arg_list[0];
        $post_vars = $arg_list[1];
        $get_vars = $arg_list[2];
        $textlab = $arg_list[3];

        $term = ($get_vars["cuitagged"]?$get_vars["cuitagged"]:$post_vars["cuitagged"]);
        if (isset($get_vars["phr"]) && $get_vars["phr"]) {
            $text = ($textlab["phrased_text"]?$textlab["phrased_text"]:$post_vars["phrased_text"]);
        } else {
            $text = ($textlab["cuitagged_text"]?$textlab["cuitagged_text"]:$post_vars["cuitagged_text"]);
        }
        $sequence = (isset($get_vars["seq"])?$get_vars["seq"]:$get_vars["phr"]);

        $tokens = explode(" ", $text);
        $i = 0;
        foreach($tokens as $unit) {
            $i++;
            $new_text[] = ($sequence==$i?"<b>$unit</b>":$unit)."<span style='color:#A0A0A0; font-weight: bold;'>$i</span>";
        }
        $itemized = implode(" ", $new_text);
        //return $text;
        if (isset($get_vars["phr"])) {
            return preg_replace("/\b(".str_replace("|", "\|", $term).")/", "<span class='highlight'>$1</span>", $itemized);
        } else {
            return preg_replace("/\b(".$term.")/", "<span class='highlight'>$1</span>", $itemized);
        }
    }

    function form_manage_negation() {

        $arg_list = $this->arglist;
        $menu_id = $arg_list[0];
        $post_vars = $arg_list[1];
        $get_vars = $arg_list[2];
        $textlab = $arg_list[3];
        $sequence = (isset($get_vars["seq"])?$get_vars["seq"]:$get_vars["phr"]);

        $c = new concept();

        print "<a name='cuitagged'>";
        print "<span class='newstitle'>".FTITLE_MANAGE_NEGATION."</span><br>";
        print "<span class='tinylight'>".INSTR_MANAGE_NEGATION."</span><br>";
        print "<table width='600' cellpadding='3' cellspacing='1' style='border: 1px solid black'>";
        // MANAGER FORM
        print "<form action = '".$_SERVER["PHP_SELF"]."?page=PROCESSING&menu_id=".$get_vars["menu_id"].($get_vars["document_id"]?"&document_id=".$get_vars["document_id"]:"")."&tab=".$get_vars["tab"]."&cuitagged=".$get_vars["cuitagged"]."&seq=$sequence".(isset($get_vars["text"])?"&text=".$get_vars["text"]:"")."#cuidetail' name='form_manage_cui' method='post'>";
        //
        print "<tr valign='top'><td colspan='4'>";

        $cuitagged = ($get_vars["cuitagged"]?$get_vars["cuitagged"]:$post_vars["cuitagged"]);
        list($input_term,$input_tag,$input_cui) = explode("_", $cuitagged);

        // extra processing if phrase
        if (isset($get_vars["phr"])) {
            // normalize phrase
            $phrase_tokens = explode("|", $input_term);
            sort($phrase_tokens);
            $input_term = implode(" ", $phrase_tokens);
        }
        if (isset($get_vars["phr"])) {
            print "<div id='phrase' style='visibility:show;'>";
            print $this->mark_tagged($cuitagged, $textlab["phrased_text"], $get_vars);
            print "</div>";
        } else {
            print "<div id='cuitagged' style='visibility: show;'>";
            print $this->mark_tagged($cuitagged, $textlab["cuitagged_text"], $get_vars);
            print "</div>";
        }

        print "</td></tr>";
        print "<tr valign='top' bgcolor='#99CCFF'><td colspan='4'>";
        print "UMLS LEXICON LOOKUP<br>";
        print "</td></tr>";
        print "<tr valign='top' bgcolor='white'><td colspan='4'>";
        print umlstools::lexicon_lookup($input_term);
        print "</td></tr>";
        print "<tr valign='top' bgcolor='#99CCFF'><td colspan='4'>";
        print "WORDNET LOOKUP<br>";
        print "</td></tr>";
        print "<tr valign='top' bgcolor='white'><td colspan='4' class='tabletext'>";
        print umlstools::wordnet_lookup($input_term);
        print "</td></tr>";
        print "<tr valign='top'><td colspan='4'>";
        print "<span class='error'>".($get_vars["cuitagged"]?$get_vars["cuitagged"]:$post_vars["cuitagged"]).": <font color='red'><b>$sequence</b></font></span><br> ";
        print "</td></tr>";
        print "<tr valign='top' bgcolor='#99CCFF'><td colspan='4'>";
        print "CONCEPT LOOKUP RESULTS<br>";
        print "</td></tr>";
        print "<tr valign='top'><td colspan='4'>";
        print "<a name='cuidetail'><br>";
        //print "<input type='radio' name='query_format' ".($post_vars["query_format"]=="NS"?"checked":"")." onchange='javascript:submit();' value='NS'> Normalized string<br>";
        //print "<input type='radio' name='query_format' ".($post_vars["query_format"]=="NN"?"checked":"")." onchange='javascript:submit();' value='NN'> Convert to noun<br>";
        $baseform = $c->baseform($input_term);
        $searchterm = ($baseform?$baseform:$input_term);
        $list_cached = $c->get_list_cached_cui($searchterm);
        $concept_list = array();
        if (is_array($list_cached)) {
            $concept_list = array_merge($concept_list, $list_cached);
        }
        $list_umls_ns = $c->get_list_umls_nscui($searchterm);
        if (is_array($list_umls_ns)) {
            $concept_list = array_merge($concept_list, $list_umls_ns);
        }
        $list_umls_cs = $c->get_list_umls_cscui($searchterm);
        if (is_array($list_umls_cs)) {
            $concept_list = array_merge($concept_list, $list_umls_cs);
        }
        $list_umls_norm = $c->get_list_umls_norm($searchterm);
        if (is_array($list_umls_norm)) {
            $concept_list = array_merge($concept_list, $list_umls_norm);
        }
        //$concept_list = array_merge($list_cached, $list_umls_ns, $list_umls_cs, $list_umls_norm);
        $concept_list = array_unique($concept_list);
        if (count($concept_list)==0) {
            $concept_list = array($input_cui);
        }
        print $this->corpus_list($input_term, $get_vars["document_id"]);
        foreach ($concept_list as $concept) {
            unset($checked_override);
            $sql = "select s.NSTR, c.CUI, c.AUI, c.SAB, v.SON, c.CODE, c.STR, t.TUI, t.STY ".
                   "from umls.MRXNS_ENG s, umls.MRCONSO c, umls.MRSTY t, umls.MRSAB v ".
                   "where c.CUI =s.CUI and s.CUI = t.CUI and c.SAB = v.RSAB and c.ISPREF = 'Y' ".
                   "and c.LAT = 'ENG' and c.CUI = '$concept' ".
                   "group by c.SAB, t.STY, c.CODE, c.STR";
            if ($result = mysql_query($sql)) {
                if (mysql_num_rows($result)) {
                    print "<table width='600' bgcolor='white' cellpadding='2' cellspacing='0'>";
                    print "<tr valign='top' bgcolor='#CCFF99'>".
                          "<td class='tabletext'>SAB</td>".
                          "<td class='tabletext'>CODE</td>".
                          "<td class='tabletext'>STR</td>".
                          "<td class='tabletext'>STY</td></tr>";
                    $prev_cui = "";
                    while(list($nstr, $cui, $aui, $sab, $son, $code, $str, $tui, $sty) = mysql_fetch_array($result)) {
                        if ($prev_cui <> $cui) {
                            $distance = 1/($c->corpus_semantic_distance($cui));
                            $bigram_prob = $c->corpus_bigram_probability($cui);
                            $left_token = new token($this->token_by_position($sequence-1, $textlab["cuitagged_text"]));
                            $left_tag = $left_token->get_tag();
                            $right_token = new token($this->token_by_position($sequence+1, $textlab["cuitagged_text"]));
                            $right_tag = $right_token->get_tag();
                            $pos_prob = $c->corpus_pos_probability($concept, $input_term, $input_tag, $left_tag, $right_tag);
                            $weight = "<font color='red'>".($distance * $bigram_prob * $pos_prob)."</font>";
                            $checked = ($input_cui==$cui?"checked":"");
                            print "<tr valign='top' bgcolor='#FFCC33'>".
                                  "<td colspan='4'><input type='radio' name='cui' $checked value='$cui'>".
                                  "<span class='service'>$cui</span> ".$weight."</td></tr>";
                            print "<tr valign='top' bgcolor='white'><td colspan='4'>".umlstools::mrdef_lookup($cui)."</td></tr>";
                        }
                        print "<tr valign='top' bgcolor='white'>".
                              "<td class='tabletext'><font color='blue'><a name='$sab' title='$son'>$sab</a></font></td>".
                              "<td class='tabletext'>$code</td>".
                              "<td class='tabletext'>$str</td>".
                              "<td class='tabletext' nowrap><b>$tui</b> $sty</td></tr>";
                        $prev_cui = $cui;
                    }
                    print "</table><br>";
                    print "<input type='hidden' name='document_id' value='".($get_vars["document_id"]?$get_vars["document_id"]:$textlab["document_id"])."'>";
                    $checked_override = "";
                } else {
                    $checked_override = "checked";
                }
            }
        }
        print "</td></tr>";
        print "<tr valign='top' bgcolor='#FFCC33'>".
              "<td colspan='4'><input type='radio' name='cui' ".($input_cui==="C0000000"?"checked":"")." value='C0000000'>".
              "<span class='service'>C0000000 (NO CUI)</span></td></tr>";
        $mytoken = new token($get_vars["cuitagged"]);
        $mycui = $mytoken->get_cui();
        print "<tr valign='top' bgcolor='#FFCC33'>".
              "<td colspan='4'>".
              "<input type='radio' name='cui' value=''>".
              "<span class='service'>OVERRIDE CUI</span> ".
              "<input type='text' class='textbox' size='8' maxlength='8' name='override' $checked_override value='".($checked_override?$mycui:"")."' style='border: 1px solid #000000'>".
              "</td></tr>";
        print "<tr valign='top' bgcolor='white'><td colspan='4'>";
        print tagger::treebank_select($input_tag);
        print "<span class='boxtitle'>".LBL_CHANGE_TYPE."</span><br> ";
        print "<span class='tinylight'>".INSTR_CHANGE_TYPE."</span><br>";
        print "<span class='service'>";
        print "<input type='radio' class='textbox' name='change_type' value='item' checked > ITEM-SPECIFIC <br>";
        print "<input type='radio' class='textbox' name='change_type' value='global' > GLOBAL <br>";
        print "</span><br>";
        print "<table cellpadding='2' cellspacing='1'>";
        print "<tr><td>";
        print "<input type='hidden' name='input_term' value='".$input_term."'>";
        print "<input type='hidden' name='cuitagged' value='".(isset($get_vars["cuitagged"])?$get_vars["cuitagged"]:$post_vars["cuitagged"])."'>";
        print "<input type='hidden' name='cuitagged_text' value='".(isset($textlab["cuitagged_text"])?$textlab["cuitagged_text"]:$post_vars["cuitagged_text"])."'>";
        print "<input type='hidden' name='phrased_text' value='".(isset($textlab["phrased_text"])?$textlab["phrased_text"]:$post_vars["phrased_text"])."'>";
        print "<input type='hidden' name='sequence' value='".$sequence."'>";
        if (isset($get_vars["phr"])) {
            print "<input type='hidden' name='phrase_seq' value='".(isset($get_vars["phr"])?$get_vars["phr"]:$post_vars["phrase_seq"])."'>";
        }
        if (isset($get_vars["text"])) {
            print "<input type='hidden' name='text' value='".(isset($get_vars["text"])?$get_vars["text"]:$post_vars["text"])."'>";
        }
        print "<input type='submit' value = 'Adopt Configuration' class='textbox' name='submitcui' title='Changes the token in tagged text.' style='border: 1px solid #000000'> ";
        //print "<input type='submit' value = 'Previous' class='textbox' name='submitcui' title='Go to previous tagged token' style='border: 1px solid #000000'> ";
        print "<input type='submit' value = 'Next' class='textbox' name='submitcui' title='Go to next tagged token' style='border: 1px solid #000000'> ";
        print "</td></tr>";
        print "<tr><td>";
        print "<input type='submit' value = 'Cancel' class='textbox' name='submitcui' style='border: 1px solid #000000'> ";
        print "</td></tr>";
        print "</table>";

        print "</td></tr>";
        print "</form>";
        print "</table>";
    }

    function corpus_list($searchterm, $document_id) {

        $baseform = concept::baseform($searchterm);

        $sql = "select document_id, term, concept_id, treebank_id, sum(frequency) ".
               "from m_textlab_terms ".
               "where term = '$searchterm' or term = '".ucfirst($searchterm)."' or term = '".$baseform."' ".
               "and document_id <> '$document_id' ".
               "group by document_id, term, concept_id, treebank_id";
        if ($result = mysql_query($sql)) {
            $return = "";
            if (mysql_num_rows($result)) {
                $return .= "<table width='100%'>";
                $return .= "<tr valign='top'><td>";
                $return .= "<span class='boxtitle'>CONSISTENCY CHECK:</span><br>";
                $return .= "<span class='textbox'>This term is found in these other documents.&gt;&gt;</span>";
                $return .= "</td><td>";
                $return .= "<table width='300' cellpadding='2' cellspacing='1'>";
                $return .= "<tr bgcolor='#9999FF'><td nowrap class='filter'>DOC ID</td><td class='filter'>TERM</td>".
                           "<td class='filter'>CONCEPT</td><td nowrap class='filter'>TREEBANK ID</td><td class='filter'>FREQ</td></tr>";
                while (list($doc_id, $term, $concept, $tree_id, $count) = mysql_fetch_array($result)) {
                    $return .= "<tr><td class='filter'>$doc_id</td><td class='filter'>$term</td>".
                               "<td class='filter'><font color='red'>$concept</font></td>".
                               "<td class='filter'><font color='blue'>$tree_id</font></td>".
                               "<td class='filter'>$count</td></tr>";
                }
                $return .= "</table>";
                $return .= "</td></tr></table>";
                return $return;
            }
        }
    }

    function process_manage_cui() {

        $arg_list = $this->arglist;
        $menu_id = $arg_list[0];
        $post_vars = $arg_list[1];
        $get_vars = $arg_list[2];
        $textlab = $arg_list[3];

        switch($post_vars["submitcui"]) {
        case "Adopt Configuration":
            if ($post_vars["input_term"] && $post_vars["cuitagged"] && $textlab["cuitagged_text"]) {
                if ($post_vars["override"]) {
                    $new_token = $post_vars["input_term"]."_".$post_vars["treebank"].($post_vars["override"]?"_".$post_vars["override"]:"");
                } else {
                    $new_token = $post_vars["input_term"]."_".$post_vars["treebank"].($post_vars["cui"]?"_".$post_vars["cui"]:"");
                }
                if ($post_vars["phrase_seq"]) {
                    if ($post_vars["cui"]=="NOCUI") {
                        $phrased_text = $this->remove_by_tagnumber($post_vars["phrase_seq"], $get_vars["cuitagged"], $textlab["phrased_text"]);
                    } else {
                        if ($post_vars["change_type"]=="item") {
                            $phrased_text = $this->replace_by_tagnumber($post_vars["phrase_seq"], $new_token, $textlab["phrased_text"]);
                        } else {
                            $phrased_text = preg_replace("/\b(".$post_vars["cuitagged"].")\b/", "$new_token", $textlab["phrased_text"]);
                        }
                    }
                    $sql = "update m_textlab_document set ".
                           "phrased_text = '$phrased_text' ".
                           "where document_id = '".$post_vars["document_id"]."'";
                    if ($result = mysql_query($sql)) {
                        $sequence = ($post_vars["sequence"]?$post_vars["sequence"]:$post_vars["phrase_seq"]);
                        $next_token = $this->retrieve_token($sequence, $textlab["phrased_text"]);
                        header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$next_token[0]."&phr=".$next_token[1]."#cuitagged");
                        //header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$new_token."&phr=".$post_vars["sequence"]."&text=".$get_vars["text"]."#cuitagged");
                    }
                } else {
                    if ($post_vars["cui"]=="NOCUI") {
                        $cuitagged_text = $this->remove_by_tagnumber($post_vars["sequence"], $get_vars["cuitagged"], $textlab["cuitagged_text"]);
                    } else {
                        if ($post_vars["change_type"]=="item") {
                            $cuitagged_text = $this->replace_by_tagnumber($post_vars["sequence"], $new_token, $textlab["cuitagged_text"]);
                        } else {
                            $cuitagged_text = preg_replace("/\b(".$post_vars["cuitagged"].")\b/", "$new_token", $textlab["cuitagged_text"]);
                        }
                    }
                    $sql = "update m_textlab_document set ".
                           "cuitagged_text = '$cuitagged_text' ".
                           "where document_id = '".$post_vars["document_id"]."'";
                    if ($result = mysql_query($sql) or die(mysql_errno().": ".mysql_error())) {
                        $sequence = ($post_vars["sequence"]?$post_vars["sequence"]:$post_vars["phrase_seq"]);
                        $next_token = $this->retrieve_token($sequence, $textlab["cuitagged_text"]);
                        header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$next_token[0]."&seq=".$next_token[1]."&text=".$get_vars["text"]."#cuitagged");
                    }
                }
            }
            break;
        case "Next":
            print_r($post_vars);
            if (($post_vars["sequence"] || $post_vars["phrase_seq"]) && ($textlab["cuitagged_text"] || $textlab["cuitagged_text_predicted"])) {
                $sequence = ($post_vars["sequence"]?$post_vars["sequence"]:$post_vars["phrase_seq"]);
                if ($post_vars["phrase_seq"]) {
                    $next_token = $this->retrieve_token($sequence, $textlab["phrased_text"]);
                } else {
                    $next_token = $this->retrieve_token($sequence, $textlab["cuitagged_text"]);
                }
            }
            if ($post_vars["phrase_seq"]) {
                header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$next_token[0]."&phr=".$next_token[1]."#cuitagged");
            } else {
                header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$next_token[0]."&seq=".$next_token[1]."&text=".$get_vars["text"]."#cuitagged");
            }
            break;

        case "Previous":
            if (($post_vars["sequence"] || $post_vars["phrase_seq"]) && ($textlab["cuitagged_text"] || $textlab["cuitagged_text_predicted"])) {
                $sequence = ($post_vars["sequence"]?$post_vars["sequence"]:$post_vars["phrase_seq"]);
                if ($post_vars["phrase_seq"]) {
                    $previous_token = $this->retrieve_previous_token($sequence, $textlab["phrased_text"]);
                } else {
                    $previous_token = $this->retrieve_previous_token($sequence, $textlab["cuitagged_text"]);
                }
            }
            if ($post_vars["phrase_seq"]) {
                header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$previous_token[0]."&phr=".$previous_token[1]."#cuitagged");
            } else {
                header("location: ".$_SERVER["PHP_SELF"]."?page=".$get_vars["page"]."&menu_id=".$get_vars["menu_id"]."&document_id=".$post_vars["document_id"]."&tab=".$get_vars["tab"]."&cuitagged=".$previous_token[0]."&seq=".$previous_token[1]."&text=".$get_vars["text"]."#cuitagged");
            }
            break;
        default:
        }
    }

    function replace_by_tagnumber($tagnumber, $new_token, $text) {

        $tokens = explode(" ", $text);
        foreach($tokens as $item) {
            $i++;
            if ($i==$tagnumber) {
                $tokens[$i-1] = $new_token;
            }
        }
        $changed_text = implode(" ", $tokens);
        return $changed_text;
    }

    function remove_by_tagnumber($tagnumber, $new_token, $text) {

        $tokens = explode(" ", $text);
        foreach($tokens as $item) {
            $i++;
            if ($i==$tagnumber) {
                unset($tokens[$i-1]);
            }
        }
        $changed_text = implode(" ", $tokens);
        return $changed_text;
    }

    function retrieve_token($sequence, $text) {

        $tokens = explode(" ", $text);
        foreach($tokens as $item) {
            $i++;
            if ($i>$sequence) {
                if (preg_match("/(_C[0-9]{7})/", $item)) {
                    return array($tokens[$i-1],$i);
                }
            }
        }
    }

    function token_by_position($sequence, $text) {

        $tokens = explode(" ", $text);
        $i = 0;
        foreach($tokens as $item) {
            $i++;
            if ($sequence==($i+1)) {
                return $tokens[$i];
            }
        }
    }

    function retrieve_previous_token($sequence, $text) {

        $tokens = explode(" ", $text);
        foreach($tokens as $item) {
            $i++;
            if ($i>$sequence) {
                if (preg_match("/(_C[0-9]{7})/", $item)) {
                    return array($tokens[$i-2],$i);
                }
            }
        }
    }

// end
// CONCEPT MANAGER CLASS
}
?>
