paste0("Start time: ", format(Sys.time(), "%Y-%m-%d %H:%M:%S %Z", sep = "\n" ))
################################################################################
##       R PROGRAM: PSE_generate_samples.R
##
##         PROJECT: Evaluation of multiple-encounter population size
##                  estimators
##
## INVESTIGATOR(S):
##
##     DESCRIPTION: Compare estimates from Rcapture and LCMCR using
##                  simulated data.
##
##                  Part 1.  Generate samples
##
##   INPUT FILE(S): None
##
##  OUTPUT FILE(S): PSEsimSamples.Rdata
##
##      WRITTEN BY: Steve Gutreuter, CDC/CGH/DGHT Statistics, Estimation
##                                   and Modeling Team
##                  sgutreuter@cdc.gov
##
##            NOTE: Effects of sample size are revealed by varying the encounter
##                  probabilities.  Model M0 is simulated with encounter
##                  probabilties of 0.025, 0.05, 0.10, 0.15 and 0.20.
##                  Inhomogeneities in the encounter probabilities are generated
##                  from the Beta distributions having means 0.025, 0.50, 0.10,
##                  0.15, 0.20, and standard deviation equal to 0.85 x mean (See
##                  plotBetaDist.R).  The data generators for trends in encounter
##                  probabilities use length-5 vectors of encounter probabilities
##                  centered at (event 3) 0.025, 0.05, 0.10, 0.15 and 0.20.
##
##      DISCLAIMER: Although this program has been used by the Centers
##                  for Disease Control & Prevention (CDC), no warranty,
##                  expressed or implied, is made by the CDC or the U.S.
##                  Government as to the accuracy and functioning of the
##                  program and related program material nor shall the
##                  fact of distribution constitute any such warranty,
##                  and no responsibility is assumed by the CDC in
##                  connection therewith.
################################################################################
################################################################################
## Log session information
################################################################################
sessionInfo()
################################################################################
## Define file paths
################################################################################
basepath <- file.path(Sys.getenv("PROJ"), "PSE/PSEsim")
workpath <- file.path(basepath, "R")
datapath <- file.path(basepath, "data")
setwd(workpath)
################################################################################
## Source helper functions and set the RNG seed
################################################################################
source(file.path(workpath, "PSE_sim_functions.R"))
seed0 <- 321
################################################################################
## Generate random encounter histories from homogeneous (M0) populations
################################################################################
#### 10,000
M0.10K.025 <- M0_gen(10000, reps = 400, p = 0.025, K = 5, seed = seed0)
M0.10K.050 <- M0_gen(10000, reps = 400, p = 0.050, K = 5, seed = seed0)
M0.10K.100 <- M0_gen(10000, reps = 400, p = 0.100, K = 5, seed = seed0)
M0.10K.150 <- M0_gen(10000, reps = 400, p = 0.150, K = 5, seed = seed0)
M0.10K.200 <- M0_gen(10000, reps = 400, p = 0.200, K = 5, seed = seed0)
#### 1,000
M0.1K.025 <- M0_gen(1000, reps = 400, p = 0.025, K = 5, seed = seed0)
M0.1K.050 <- M0_gen(1000, reps = 400, p = 0.050, K = 5, seed = seed0)
M0.1K.100 <- M0_gen(1000, reps = 400, p = 0.100, K = 5, seed = seed0)
M0.1K.150 <- M0_gen(1000, reps = 400, p = 0.150, K = 5, seed = seed0)
M0.1K.200 <- M0_gen(1000, reps = 400, p = 0.200, K = 5, seed = seed0)
#### 25,000
M0.20K.025 <- M0_gen(20000, reps = 400, p = 0.025, K = 5, seed = seed0)
M0.20K.050 <- M0_gen(20000, reps = 400, p = 0.050, K = 5, seed = seed0)
M0.20K.100 <- M0_gen(20000, reps = 400, p = 0.100, K = 5, seed = seed0)
M0.20K.150 <- M0_gen(20000, reps = 400, p = 0.150, K = 5, seed = seed0)
M0.20K.200 <- M0_gen(20000, reps = 400, p = 0.200, K = 5, seed = seed0)
################################################################################
## Generate random encounter histories from Mh populations
################################################################################
#### 10,000
Mh.10K.025 <- Mh_gen(10000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                     seed = seed0)
Mh.10K.050 <- Mh_gen(10000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                       seed = seed0)
Mh.10K.100 <- Mh_gen(10000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                       seed = seed0)
Mh.10K.150 <- Mh_gen(10000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                     seed = seed0)
Mh.10K.200 <- Mh_gen(10000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                     seed = seed0)
#### 1,000
Mh.1K.025 <- Mh_gen(1000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                     seed = seed0)
Mh.1K.050 <- Mh_gen(1000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                       seed = seed0)
Mh.1K.100 <- Mh_gen(1000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                       seed = seed0)
Mh.1K.150 <- Mh_gen(1000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                    seed = seed0)
Mh.1K.200 <- Mh_gen(1000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                     seed = seed0)
#### 20,000
Mh.20K.025 <- Mh_gen(20000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                     seed = seed0)
Mh.20K.050 <- Mh_gen(20000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                       seed = seed0)
Mh.20K.100 <- Mh_gen(20000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                       seed = seed0)
Mh.20K.150 <- Mh_gen(20000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                     seed = seed0)
Mh.20K.200 <- Mh_gen(20000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                     seed = seed0)
################################################################################
## Generate random encounter histories from Mb populations where P(encounter)
## drops by a factor of 50% after first contact.
################################################################################
#### 10,000
Mb.10K.025  <- Mb_gen(10000, reps = 400, K = 5, p  = 0.025, frac = 0.5,
                        seed = seed0)
Mb.10K.050  <- Mb_gen(10000, reps = 400, K = 5, p  = 0.050, frac = 0.5,
                        seed = seed0)
Mb.10K.100  <- Mb_gen(10000, reps = 400, K = 5, p  = 0.100, frac = 0.5,
                        seed = seed0)
Mb.10K.150  <- Mb_gen(10000, reps = 400, K = 5, p  = 0.150, frac = 0.5,
                      seed = seed0)
Mb.10K.200  <- Mb_gen(10000, reps = 400, K = 5, p  = 0.200, frac = 0.5,
                      seed = seed0)
#### 1,000
Mb.1K.025  <- Mb_gen(1000, reps = 400, K = 5, p  = 0.025, frac = 0.5,
                        seed = seed0)
Mb.1K.050  <- Mb_gen(1000, reps = 400, K = 5, p  = 0.050, frac = 0.5,
                        seed = seed0)
Mb.1K.100  <- Mb_gen(1000, reps = 400, K = 5, p  = 0.100, frac = 0.5,
                        seed = seed0)
Mb.1K.150  <- Mb_gen(1000, reps = 400, K = 5, p  = 0.150, frac = 0.5,
                     seed = seed0)
Mb.1K.200  <- Mb_gen(1000, reps = 400, K = 5, p  = 0.200, frac = 0.5,
                      seed = seed0)
#### 20,000
Mb.20K.025  <- Mb_gen(20000, reps = 400, K = 5, p  = 0.025, frac = 0.5,
                        seed = seed0)
Mb.20K.050  <- Mb_gen(20000, reps = 400, K = 5, p  = 0.050, frac = 0.5,
                        seed = seed0)
Mb.20K.100  <- Mb_gen(20000, reps = 400, K = 5, p  = 0.100, frac = 0.5,
                        seed = seed0)
Mb.20K.150  <- Mb_gen(20000, reps = 400, K = 5, p  = 0.150, frac = 0.5,
                      seed = seed0)
Mb.20K.200  <- Mb_gen(20000, reps = 400, K = 5, p  = 0.200, frac = 0.5,
                      seed = seed0)
################################################################################
## Generate random encounter histories from Mbh populations where P(encounter)
## drops by 50% after first contact.
################################################################################
#### 10,000
Mbh.10K.025 <- Mbh_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.32448, 51.6548), frac = 0.5,
                       seed = seed0)
Mbh.10K.050 <- Mbh_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.26488, 24.0327),frac = 0.5,
                       seed = seed0)
Mbh.10K.100 <- Mbh_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.14567, 10.3111), frac = 0.5,
                       seed = seed0)
Mbh.10K.150 <- Mbh_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.02647, 5.81667), frac = 0.5,
                       seed = seed0)
Mbh.10K.200 <- Mbh_gen(10000, reps = 400, K = 5,
                       betaparms = c(0.90727, 3.62907), frac = 0.5,
                       seed = seed0)
#### 1,000
Mbh.1K.025 <- Mbh_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.32448, 51.6548), frac = 0.5,
                       seed = seed0)
Mbh.1K.050 <- Mbh_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.26488, 24.0327),frac = 0.5,
                       seed = seed0)
Mbh.1K.100 <- Mbh_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.14567, 10.3111), frac = 0.5,
                       seed = seed0)
Mbh.1K.150 <- Mbh_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.02647, 5.81667), frac = 0.5,
                      seed = seed0)
Mbh.1K.200 <- Mbh_gen(1000, reps = 400, K = 5,
                       betaparms = c(0.90727, 3.62907), frac = 0.5,
                       seed = seed0)
#### 20,000
Mbh.20K.025 <- Mbh_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.32448, 51.6548), frac = 0.5,
                       seed = seed0)
Mbh.20K.050 <- Mbh_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.26488, 24.0327),frac = 0.5,
                       seed = seed0)
Mbh.20K.100 <- Mbh_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.14567, 10.3111), frac = 0.5,
                       seed = seed0)
Mbh.20K.150 <- Mbh_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.02647, 5.81667), frac = 0.5,
                       seed = seed0)
Mbh.20K.200 <- Mbh_gen(20000, reps = 400, K = 5,
                       betaparms = c(0.90727, 3.62907), frac = 0.5,
                       seed = seed0)
################################################################################
## Generate random encounter histories from Mbht populations where P(encounter)
## drops by 50% after first contact.
################################################################################
#### 10,000
Mbht.10K.025 <- Mbht_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.32448, 51.6548), frac = 0.5,
                       seed = seed0)
Mbht.10K.050 <- Mbht_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.26488, 24.0327),frac = 0.5,
                       seed = seed0)
Mbht.10K.100 <- Mbht_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.14567, 10.3111), frac = 0.5,
                       seed = seed0)
Mbht.10K.150 <- Mbht_gen(10000, reps = 400, K = 5,
                       betaparms = c(1.02647, 5.81667), frac = 0.5,
                       seed = seed0)
Mbht.10K.200 <- Mbht_gen(10000, reps = 400, K = 5,
                       betaparms = c(0.90727, 3.62907), frac = 0.5,
                       seed = seed0)
#### 1,000
Mbht.1K.025 <- Mbht_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.32448, 51.6548), frac = 0.5,
                       seed = seed0)
Mbht.1K.050 <- Mbht_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.26488, 24.0327),frac = 0.5,
                       seed = seed0)
Mbht.1K.100 <- Mbht_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.14567, 10.3111), frac = 0.5,
                       seed = seed0)
Mbht.1K.150 <- Mbht_gen(1000, reps = 400, K = 5,
                       betaparms = c(1.02647, 5.81667), frac = 0.5,
                       seed = seed0)
Mbht.1K.200 <- Mbht_gen(1000, reps = 400, K = 5,
                       betaparms = c(0.90727, 3.62907), frac = 0.5,
                       seed = seed0)
#### 20,000
Mbht.20K.025 <- Mbht_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.32448, 51.6548), frac = 0.5,
                       seed = seed0)
Mbht.20K.050 <- Mbht_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.26488, 24.0327),frac = 0.5,
                       seed = seed0)
Mbht.20K.100 <- Mbht_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.14567, 10.3111), frac = 0.5,
                       seed = seed0)
Mbht.20K.150 <- Mbht_gen(20000, reps = 400, K = 5,
                       betaparms = c(1.02647, 5.81667), frac = 0.5,
                       seed = seed0)
Mbht.20K.200 <- Mbht_gen(20000, reps = 400, K = 5,
                       betaparms = c(0.90727, 3.62907), frac = 0.5,
                       seed = seed0)
################################################################################
## Generate random encounter histories from Mt populations
################################################################################
#### 10,000
Mt.10K.025  <- Mt_gen(10000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                      seed = seed0)
Mt.10K.050  <- Mt_gen(10000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                      seed = seed0)
Mt.10K.100  <- Mt_gen(10000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                      seed = seed0)
Mt.10K.150  <- Mt_gen(10000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                      seed = seed0)
Mt.10K.200  <- Mt_gen(10000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                      seed = seed0)
#### 1,000
Mt.1K.025  <- Mt_gen(1000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                      seed = seed0)
Mt.1K.050  <- Mt_gen(1000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                      seed = seed0)
Mt.1K.100  <- Mt_gen(1000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                      seed = seed0)
Mt.1K.150  <- Mt_gen(1000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                     seed = seed0)
Mt.1K.200  <- Mt_gen(1000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                      seed = seed0)
#### 20,000
Mt.20K.025  <- Mt_gen(20000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                      seed = seed0)
Mt.20K.050  <- Mt_gen(20000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                      seed = seed0)
Mt.20K.100  <- Mt_gen(20000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                      seed = seed0)
Mt.20K.150  <- Mt_gen(20000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                      seed = seed0)
Mt.20K.200  <- Mt_gen(20000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                      seed = seed0)
################################################################################
## Generate random encounter histories from Mht populations
################################################################################
#### 10,000
Mht.10K.025  <- Mht_gen(10000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                        seed = seed0)
Mht.10K.050  <- Mht_gen(10000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                        seed = seed0)
Mht.10K.100  <- Mht_gen(10000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                        seed = seed0)
Mht.10K.150  <- Mht_gen(10000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                        seed = seed0)
Mht.10K.200  <- Mht_gen(10000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                        seed = seed0)
#### 1,000
Mht.1K.025  <- Mht_gen(1000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                        seed = seed0)
Mht.1K.050  <- Mht_gen(1000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                        seed = seed0)
Mht.1K.100  <- Mht_gen(1000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                        seed = seed0)
Mht.1K.150  <- Mht_gen(1000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                       seed = seed0)
Mht.1K.200  <- Mht_gen(1000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                        seed = seed0)
#### 20,000
Mht.20K.025  <- Mht_gen(20000, reps = 400, K = 5, betaparms = c(1.32448, 51.6548),
                        seed = seed0)
Mht.20K.050  <- Mht_gen(20000, reps = 400, K = 5, betaparms = c(1.26488, 24.0327),
                        seed = seed0)
Mht.20K.100  <- Mht_gen(20000, reps = 400, K = 5, betaparms = c(1.14567, 10.3111),
                        seed = seed0)
Mht.20K.150  <- Mht_gen(20000, reps = 400, K = 5, betaparms = c(1.02647, 5.81667),
                        seed = seed0)
Mht.20K.200  <- Mht_gen(20000, reps = 400, K = 5, betaparms = c(0.90727, 3.62907),
                        seed = seed0)
################################################################################
## Save the generated samples
################################################################################
save(  M0.10K.025,    M0.10K.050,    M0.10K.100,    M0.10K.150,    M0.10K.200,
       Mh.10K.025,    Mh.10K.050,    Mh.10K.100,    Mh.10K.150,    Mh.10K.200,
       Mb.10K.025,    Mb.10K.050,    Mb.10K.100,    Mb.10K.150,    Mb.10K.200,
       Mt.10K.025,    Mt.10K.050,    Mt.10K.100,    Mt.10K.150,    Mt.10K.200,
      Mbh.10K.025,   Mbh.10K.050,   Mbh.10K.100,   Mbh.10K.150,   Mbh.10K.200,
      Mht.10K.025,   Mht.10K.050,   Mht.10K.100,   Mht.10K.150,   Mht.10K.200,
     Mbht.10K.025,  Mbht.10K.050,  Mbht.10K.100,  Mbht.10K.150,  Mbht.10K.200,
        M0.1K.025,     M0.1K.050,     M0.1K.100,     M0.1K.150,     M0.1K.200,
        Mh.1K.025,     Mh.1K.050,     Mh.1K.100,     Mh.1K.150,     Mh.1K.200,
        Mb.1K.025,     Mb.1K.050,     Mb.1K.100,     Mb.1K.150,     Mb.1K.200,
        Mt.1K.025,     Mt.1K.050,     Mt.1K.100,     Mt.1K.150,     Mt.1K.200,
       Mbh.1K.025,    Mbh.1K.050,    Mbh.1K.100,    Mbh.1K.150,    Mbh.1K.200,
       Mht.1K.025,    Mht.1K.050,    Mht.1K.100,    Mht.1K.150,    Mht.1K.200,
      Mbht.1K.025,   Mbht.1K.050,   Mbht.1K.100,   Mbht.1K.150,   Mbht.1K.200,
       M0.20K.025,    M0.20K.050,    M0.20K.100,    M0.20K.150,    M0.20K.200,
       Mh.20K.025,    Mh.20K.050,    Mh.20K.100,    Mh.20K.150,    Mh.20K.200,
       Mb.20K.025,    Mb.20K.050,    Mb.20K.100,    Mb.20K.150,    Mb.20K.200,
       Mt.20K.025,    Mt.20K.050,    Mt.20K.100,    Mt.20K.150,    Mt.20K.200,
      Mbh.20K.025,   Mbh.20K.050,   Mbh.20K.100,   Mbh.20K.150,   Mbh.20K.200,
      Mht.20K.025,   Mht.20K.050,   Mht.20K.100,   Mht.20K.150,   Mht.20K.200,
     Mbht.20K.025,  Mbht.20K.050,  Mbht.20K.100,  Mbht.20K.150,  Mbht.20K.200,
     file = file.path(datapath, "PSEsimSamples.Rdata"))
################################   END of FILE   ################################
paste0("Finish time: ", format(Sys.time(), "%Y-%m-%d %H:%M:%S %Z", sep = "\n" ))
